<?php 
require_once 'config.php';
session_start();

// Check if user is logged in, if not redirect to login
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'] ?? 'User';

// Handle Profile Updates
$profile_message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'update_profile') {
        $name = isset($_POST['name']) ? trim($_POST['name']) : '';
        $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
        
        if (!empty($name) && !empty($phone)) {
            $update_query = "UPDATE users SET name = ?, phone = ? WHERE id = ?";
            $stmt = $conn->prepare($update_query);
            if ($stmt) {
                $stmt->bind_param('ssi', $name, $phone, $user_id);
                if ($stmt->execute()) {
                    $_SESSION['user_name'] = $name;
                    $profile_message = '<div class="alert alert-success">Profile updated successfully!</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error updating profile.</div>';
                }
                $stmt->close();
            }
        }
    } elseif ($_POST['action'] === 'update_password') {
        $old_password = isset($_POST['old_password']) ? $_POST['old_password'] : '';
        $new_password = isset($_POST['new_password']) ? $_POST['new_password'] : '';
        $confirm_password = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';
        
        // Verify old password
        $user_query = "SELECT password FROM users WHERE id = ?";
        $stmt = $conn->prepare($user_query);
        if ($stmt) {
            $stmt->bind_param('i', $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            
            if ($user && password_verify($old_password, $user['password'])) {
                if ($new_password === $confirm_password && strlen($new_password) >= 6) {
                    $hashed = password_hash($new_password, PASSWORD_DEFAULT);
                    $pwd_query = "UPDATE users SET password = ? WHERE id = ?";
                    $pwd_stmt = $conn->prepare($pwd_query);
                    if ($pwd_stmt) {
                        $pwd_stmt->bind_param('si', $hashed, $user_id);
                        if ($pwd_stmt->execute()) {
                            $profile_message = '<div class="alert alert-success">Password changed successfully!</div>';
                        }
                        $pwd_stmt->close();
                    }
                } else {
                    $profile_message = '<div class="alert alert-danger">Passwords do not match or are too short (min 6 chars).</div>';
                }
            } else {
                $profile_message = '<div class="alert alert-danger">Old password is incorrect.</div>';
            }
            $stmt->close();
        }
    } elseif ($_POST['action'] === 'submit_feedback') {
        $rating = isset($_POST['rating']) ? intval($_POST['rating']) : 0;
        $feedback = isset($_POST['feedback']) ? trim($_POST['feedback']) : '';
        
        if ($rating > 0 && !empty($feedback)) {
            $feedback_query = "INSERT INTO reviews (user_id, booking_id, rating, review_text, status) VALUES (?, NULL, ?, ?, 'pending')";
            $stmt = $conn->prepare($feedback_query);
            if ($stmt) {
                $stmt->bind_param('iis', $user_id, $rating, $feedback);
                if ($stmt->execute()) {
                    $profile_message = '<div class="alert alert-success">Feedback submitted successfully! Admin will review it.</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error submitting feedback.</div>';
                }
                $stmt->close();
            }
        } else {
            $profile_message = '<div class="alert alert-danger">Please fill all fields.</div>';
        }
    } elseif ($_POST['action'] === 'submit_contact') {
        $contact_subject = isset($_POST['contact_subject']) ? trim($_POST['contact_subject']) : '';
        $contact_message = isset($_POST['contact_message']) ? trim($_POST['contact_message']) : '';
        
        if (!empty($contact_subject) && !empty($contact_message)) {
            $contact_query = "INSERT INTO contact_messages (name, email, phone, subject, message, status) VALUES (?, ?, ?, ?, ?, 'new')";
            $stmt = $conn->prepare($contact_query);
            if ($stmt) {
                $user_email = $_SESSION['user_email'] ?? '';
                $user_phone = '';
                // Get user phone
                $phone_query = "SELECT phone FROM users WHERE id = ?";
                $phone_stmt = $conn->prepare($phone_query);
                if ($phone_stmt) {
                    $phone_stmt->bind_param('i', $user_id);
                    $phone_stmt->execute();
                    $phone_result = $phone_stmt->get_result();
                    if ($phone_row = $phone_result->fetch_assoc()) {
                        $user_phone = $phone_row['phone'];
                    }
                    $phone_stmt->close();
                }
                
                $stmt->bind_param('sssss', $user_name, $user_email, $user_phone, $contact_subject, $contact_message);
                if ($stmt->execute()) {
                    $profile_message = '<div class="alert alert-success">Contact message sent to admin! Admin will respond soon.</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error sending contact message.</div>';
                }
                $stmt->close();
            }
        }
    } elseif ($_POST['action'] === 'book_service') {
        $service_id = isset($_POST['service_id']) ? intval($_POST['service_id']) : 0;
        $booking_date = isset($_POST['booking_date']) ? $_POST['booking_date'] : '';
        $booking_time = isset($_POST['booking_time']) ? $_POST['booking_time'] : '';
        
        if ($service_id > 0 && !empty($booking_date) && !empty($booking_time)) {
            $booking_query = "INSERT INTO bookings (user_id, service_id, booking_date, booking_time, status) VALUES (?, ?, ?, ?, 'pending')";
            $stmt = $conn->prepare($booking_query);
            if ($stmt) {
                $stmt->bind_param('iiss', $user_id, $service_id, $booking_date, $booking_time);
                if ($stmt->execute()) {
                    $profile_message = '<div class="alert alert-success">Service booked successfully! Admin will confirm soon.</div>';
                } else {
                    $profile_message = '<div class="alert alert-danger">Error booking service.</div>';
                }
                $stmt->close();
            }
        }
    }
}

// Fetch user profile data
$user_query = "SELECT name, email, phone FROM users WHERE id = ?";
$user_stmt = $conn->prepare($user_query);
$user_stmt->bind_param('i', $user_id);
$user_stmt->execute();
$user_result = $user_stmt->get_result();
$user_data = $user_result->fetch_assoc();
$user_stmt->close();

// Fetch services for booking
$services_query = "SELECT id, name, category, price FROM services WHERE status = 'active' ORDER BY name ASC";
$services_result = $conn->query($services_query);
$services = array();
if ($services_result && $services_result->num_rows > 0) {
    while ($row = $services_result->fetch_assoc()) {
        $services[] = $row;
    }
}

// Fetch user bookings
$bookings_query = "SELECT b.id, s.name as service_name, b.booking_date, b.booking_time, b.status FROM bookings b JOIN services s ON b.service_id = s.id WHERE b.user_id = ? ORDER BY b.created_at DESC";
$bookings_stmt = $conn->prepare($bookings_query);
$bookings_stmt->bind_param('i', $user_id);
$bookings_stmt->execute();
$bookings_result = $bookings_stmt->get_result();
$bookings = array();
if ($bookings_result && $bookings_result->num_rows > 0) {
    while ($row = $bookings_result->fetch_assoc()) {
        $bookings[] = $row;
    }
}
$bookings_stmt->close();

// Fetch user feedback
$feedback_query = "SELECT id, rating, review_text, status, created_at FROM reviews WHERE user_id = ? ORDER BY created_at DESC";
$feedback_stmt = $conn->prepare($feedback_query);
$feedback_stmt->bind_param('i', $user_id);
$feedback_stmt->execute();
$feedback_result = $feedback_stmt->get_result();
$user_feedback = array();
if ($feedback_result && $feedback_result->num_rows > 0) {
    while ($row = $feedback_result->fetch_assoc()) {
        $user_feedback[] = $row;
    }
}
$feedback_stmt->close();

// Fetch contact messages
$contact_query = "SELECT id, subject, message, status, created_at FROM contact_messages WHERE name = ? OR email = ? ORDER BY created_at DESC";
$contact_stmt = $conn->prepare($contact_query);
$contact_stmt->bind_param('ss', $user_name, $user_data['email']);
$contact_stmt->execute();
$contact_result = $contact_stmt->get_result();
$contact_messages = array();
if ($contact_result && $contact_result->num_rows > 0) {
    while ($row = $contact_result->fetch_assoc()) {
        $contact_messages[] = $row;
    }
}
$contact_stmt->close();
?><!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Customer Dashboard | Cleaning Service</title>
    <link rel="stylesheet" href="assets/css/bootstrap.min.css">
    <link rel="stylesheet" href="assets/css/fontawesome.min.css">
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        :root {
            --primary-color: #109c3d;
            --primary-dark: #0f8c3d;
            --light-bg: #f8fbff;
            --border-color: rgba(16, 156, 61, 0.2);
        }

        * {
            transition: all 0.3s ease;
        }

        body {
            background: linear-gradient(135deg, #f8fbff, #f1f5f4);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        /* Sidebar Navigation */
        .dashboard-wrapper {
            display: grid;
            grid-template-columns: 280px 1fr;
            gap: 0;
            min-height: 100vh;
        }

        .sidebar {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            padding: 30px 20px;
            position: sticky;
            top: 0;
            height: 100vh;
            overflow-y: auto;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        .sidebar-brand {
            font-weight: 700;
            font-size: 18px;
            margin-bottom: 30px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .sidebar-menu {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .sidebar-menu li {
            margin-bottom: 10px;
        }

        .sidebar-menu a {
            color: rgba(255, 255, 255, 0.8);
            text-decoration: none;
            padding: 12px 15px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            gap: 12px;
            font-size: 14px;
        }

        .sidebar-menu a:hover,
        .sidebar-menu a.active {
            background: rgba(255, 255, 255, 0.2);
            color: white;
        }

        .sidebar-menu i {
            width: 20px;
            text-align: center;
        }

        .logout-btn {
            margin-top: 30px;
            padding-top: 30px;
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        /* Main Content */
        .main-content {
            padding: 30px;
            overflow-y: auto;
        }

        .header-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 40px;
            flex-wrap: wrap;
            gap: 20px;
        }

        .header-section h1 {
            font-size: 32px;
            font-weight: 700;
            color: #0f172a;
        }

        .user-profile-mini {
            display: flex;
            align-items: center;
            gap: 15px;
            background: white;
            padding: 15px 20px;
            border-radius: 12px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }

        .avatar {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: var(--primary-color);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 700;
            font-size: 18px;
        }

        .user-info {
            display: flex;
            flex-direction: column;
        }

        .user-info p {
            margin: 0;
            font-size: 12px;
            color: #64748b;
        }

        .user-info strong {
            color: #0f172a;
            font-size: 14px;
        }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }

        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.08);
            border-left: 5px solid var(--primary-color);
        }

        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--primary-color);
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 14px;
            color: #64748b;
            font-weight: 500;
        }

        .stat-icon {
            float: right;
            font-size: 32px;
            color: rgba(16, 156, 61, 0.1);
        }

        /* Content Sections */
        .content-section {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.08);
            margin-bottom: 30px;
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            flex-wrap: wrap;
            gap: 15px;
        }

        .section-header h2 {
            font-size: 22px;
            font-weight: 700;
            color: #0f172a;
        }

        .btn-primary-custom {
            background: var(--primary-color);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-primary-custom:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(16, 156, 61, 0.3);
        }

        /* Bookings Table */
        .bookings-table {
            overflow-x: auto;
        }

        .bookings-table table {
            width: 100%;
            border-collapse: collapse;
        }

        .bookings-table thead {
            background: var(--light-bg);
        }

        .bookings-table th {
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #0f172a;
            border-bottom: 2px solid var(--border-color);
        }

        .bookings-table td {
            padding: 15px;
            border-bottom: 1px solid var(--border-color);
            color: #64748b;
        }

        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        .status-pending {
            background: rgba(251, 191, 36, 0.1);
            color: #d97706;
        }

        .status-confirmed {
            background: rgba(59, 130, 246, 0.1);
            color: #2563eb;
        }

        .status-completed {
            background: rgba(16, 156, 61, 0.1);
            color: var(--primary-color);
        }

        .status-cancelled {
            background: rgba(239, 68, 68, 0.1);
            color: #ef4444;
        }

        /* Profile Form */
        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            font-weight: 600;
            color: #0f172a;
            margin-bottom: 8px;
            display: block;
        }

        .form-control {
            width: 100%;
            padding: 12px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            font-size: 14px;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(16, 156, 61, 0.1);
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 50px 20px;
        }

        .empty-state-icon {
            font-size: 60px;
            color: rgba(16, 156, 61, 0.2);
            margin-bottom: 20px;
        }

        .empty-state-title {
            font-size: 18px;
            font-weight: 700;
            color: #0f172a;
            margin-bottom: 10px;
        }

        .empty-state-text {
            color: #64748b;
            margin-bottom: 25px;
        }

        /* Tabs */
        .nav-tabs {
            display: flex;
            gap: 10px;
            border-bottom: 2px solid var(--border-color);
            margin-bottom: 25px;
        }

        .nav-tab {
            background: none;
            border: none;
            color: #64748b;
            padding: 12px 20px;
            cursor: pointer;
            font-weight: 600;
            border-bottom: 3px solid transparent;
            transition: all 0.3s ease;
        }

        .nav-tab.active {
            color: var(--primary-color);
            border-bottom-color: var(--primary-color);
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .dashboard-wrapper {
                grid-template-columns: 1fr;
            }

            .sidebar {
                position: fixed;
                left: -280px;
                width: 280px;
                height: 100vh;
                z-index: 1000;
                transition: left 0.3s ease;
            }

            .sidebar.active {
                left: 0;
            }

            .main-content {
                padding: 20px;
            }

            .header-section h1 {
                font-size: 24px;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }

            .content-section {
                padding: 20px;
            }

            .bookings-table {
                font-size: 12px;
            }

            .bookings-table th,
            .bookings-table td {
                padding: 10px;
            }

            .toggle-sidebar {
                display: inline-block;
                background: var(--primary-color);
                color: white;
                border: none;
                padding: 10px 15px;
                border-radius: 8px;
                cursor: pointer;
                font-size: 16px;
                margin-bottom: 20px;
            }

            .section-header {
                flex-direction: column;
                align-items: flex-start;
            }

            .user-profile-mini {
                flex-direction: column;
                align-items: center;
                text-align: center;
                width: 100%;
            }
        }

        @media (max-width: 480px) {
            .header-section h1 {
                font-size: 20px;
            }

            .stat-card {
                padding: 15px;
            }

            .stat-value {
                font-size: 24px;
            }

            .section-header h2 {
                font-size: 18px;
            }

            .content-section {
                padding: 15px;
            }

            .nav-tab {
                padding: 10px 15px;
                font-size: 12px;
            }
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <!-- Sidebar Navigation -->
        <div class="sidebar" id="sidebar">
            <div class="sidebar-brand">
                <i class="fa-solid fa-home-user"></i>
                Dashboard
            </div>
            <ul class="sidebar-menu">
                <li><a href="#overview" class="menu-link active" data-tab="overview"><i class="fa-solid fa-chart-line"></i> Overview</a></li>
                <li><a href="#bookings" class="menu-link" data-tab="bookings"><i class="fa-solid fa-calendar-check"></i> My Bookings</a></li>
                <li><a href="#history" class="menu-link" data-tab="history"><i class="fa-solid fa-history"></i> History</a></li>
                <li><a href="#profile" class="menu-link" data-tab="profile"><i class="fa-solid fa-user"></i> Profile</a></li>
                <li><a href="#reviews" class="menu-link" data-tab="reviews"><i class="fa-solid fa-star"></i> Reviews</a></li>
                <li class="logout-btn"><a href="logout.php"><i class="fa-solid fa-sign-out-alt"></i> Logout</a></li>
            </ul>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <!-- Toggle Sidebar Button -->
            <button class="toggle-sidebar" id="toggleBtn" style="display:none;">
                <i class="fa-solid fa-bars"></i>
            </button>

            <!-- Header Section -->
            <div class="header-section">
                <div>
                    <h1 id="pageTitle">Welcome Back!</h1>
                </div>
                <div class="user-profile-mini">
                    <div class="avatar"><?php echo strtoupper(substr($_SESSION['user_name'], 0, 1)); ?></div>
                    <div class="user-info">
                        <strong><?php echo htmlspecialchars($_SESSION['user_name']); ?></strong>
                        <p>Premium Member</p>
                    </div>
                </div>
            </div>

            <!-- TAB 1: Overview -->
            <div id="overview" class="tab-content active">
                <!-- Stats Grid -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <i class="fa-solid fa-calendar-check stat-icon"></i>
                        <div class="stat-value">8</div>
                        <div class="stat-label">Total Bookings</div>
                    </div>
                    <div class="stat-card">
                        <i class="fa-solid fa-check-circle stat-icon"></i>
                        <div class="stat-value">7</div>
                        <div class="stat-label">Completed Services</div>
                    </div>
                    <div class="stat-card">
                        <i class="fa-solid fa-hourglass stat-icon"></i>
                        <div class="stat-value">1</div>
                        <div class="stat-label">Pending Services</div>
                    </div>
                    <div class="stat-card">
                        <i class="fa-solid fa-star stat-icon"></i>
                        <div class="stat-value">4.8</div>
                        <div class="stat-label">Average Rating</div>
                    </div>
                </div>

                <!-- Recent Bookings -->
                <div class="content-section">
                    <div class="section-header">
                        <h2>Recent Bookings</h2>
                        <button class="btn-primary-custom">New Booking</button>
                    </div>
                    <div class="bookings-table">
                        <table>
                            <thead>
                                <tr>
                                    <th>Service</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th>Amount</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Home Deep Cleaning</td>
                                    <td>Feb 5, 2026</td>
                                    <td><span class="status-badge status-confirmed">Confirmed</span></td>
                                    <td>$150</td>
                                    <td><a href="#" style="color: var(--primary-color); text-decoration: none;">View</a></td>
                                </tr>
                                <tr>
                                    <td>Window Cleaning</td>
                                    <td>Feb 3, 2026</td>
                                    <td><span class="status-badge status-completed">Completed</span></td>
                                    <td>$80</td>
                                    <td><a href="#" style="color: var(--primary-color); text-decoration: none;">View</a></td>
                                </tr>
                                <tr>
                                    <td>Office Cleaning</td>
                                    <td>Jan 28, 2026</td>
                                    <td><span class="status-badge status-completed">Completed</span></td>
                                    <td>$200</td>
                                    <td><a href="#" style="color: var(--primary-color); text-decoration: none;">View</a></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Quick Stats -->
                <div class="content-section">
                    <h2 style="margin-bottom: 20px;">Spending Overview</h2>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px;">
                        <div style="padding: 15px; background: var(--light-bg); border-radius: 8px; text-align: center;">
                            <div style="font-size: 24px; font-weight: 700; color: var(--primary-color);">$950</div>
                            <div style="font-size: 12px; color: #64748b; margin-top: 5px;">Total Spent</div>
                        </div>
                        <div style="padding: 15px; background: var(--light-bg); border-radius: 8px; text-align: center;">
                            <div style="font-size: 24px; font-weight: 700; color: var(--primary-color);">$150</div>
                            <div style="font-size: 12px; color: #64748b; margin-top: 5px;">This Month</div>
                        </div>
                        <div style="padding: 15px; background: var(--light-bg); border-radius: 8px; text-align: center;">
                            <div style="font-size: 24px; font-weight: 700; color: var(--primary-color);">$120</div>
                            <div style="font-size: 12px; color: #64748b; margin-top: 5px;">Average Service</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- TAB 2: My Bookings -->
            <div id="bookings" class="tab-content">
                <div class="content-section">
                    <div class="section-header">
                        <h2>My Bookings</h2>
                        <button class="btn-primary-custom">New Booking</button>
                    </div>

                    <!-- Filter Tabs -->
                    <div class="nav-tabs">
                        <button class="nav-tab active" onclick="filterBookings('all')">All</button>
                        <button class="nav-tab" onclick="filterBookings('pending')">Pending</button>
                        <button class="nav-tab" onclick="filterBookings('confirmed')">Confirmed</button>
                        <button class="nav-tab" onclick="filterBookings('completed')">Completed</button>
                    </div>

                    <div class="bookings-table">
                        <table>
                            <thead>
                                <tr>
                                    <th>Booking ID</th>
                                    <th>Service</th>
                                    <th>Date & Time</th>
                                    <th>Status</th>
                                    <th>Amount</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>#BK001</td>
                                    <td>Home Deep Cleaning</td>
                                    <td>Feb 5, 2026 - 10:00 AM</td>
                                    <td><span class="status-badge status-confirmed">Confirmed</span></td>
                                    <td>$150</td>
                                    <td>
                                        <a href="#" style="color: var(--primary-color); text-decoration: none; margin-right: 10px;">Edit</a>
                                        <a href="#" style="color: #ef4444; text-decoration: none;">Cancel</a>
                                    </td>
                                </tr>
                                <tr>
                                    <td>#BK002</td>
                                    <td>Window Cleaning</td>
                                    <td>Feb 3, 2026 - 02:00 PM</td>
                                    <td><span class="status-badge status-completed">Completed</span></td>
                                    <td>$80</td>
                                    <td>
                                        <a href="#" style="color: var(--primary-color); text-decoration: none;">Review</a>
                                    </td>
                                </tr>
                                <tr>
                                    <td>#BK003</td>
                                    <td>Carpet Cleaning</td>
                                    <td>Jan 15, 2026 - 11:00 AM</td>
                                    <td><span class="status-badge status-completed">Completed</span></td>
                                    <td>$120</td>
                                    <td>
                                        <a href="#" style="color: var(--primary-color); text-decoration: none;">Review</a>
                                    </td>
                                </tr>
                                <tr>
                                    <td>#BK004</td>
                                    <td>Kitchen Deep Clean</td>
                                    <td>Jan 8, 2026 - 09:00 AM</td>
                                    <td><span class="status-badge status-completed">Completed</span></td>
                                    <td>$100</td>
                                    <td>
                                        <a href="#" style="color: var(--primary-color); text-decoration: none;">Review</a>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- TAB 3: History -->
            <div id="history" class="tab-content">
                <div class="content-section">
                    <div class="section-header">
                        <h2>Service History</h2>
                    </div>

                    <div style="display: flex; flex-direction: column; gap: 15px;">
                        <div style="padding: 20px; background: var(--light-bg); border-left: 4px solid var(--primary-color); border-radius: 8px;">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 10px;">
                                <strong style="color: #0f172a;">Home Deep Cleaning</strong>
                                <span style="background: rgba(16, 156, 61, 0.1); color: var(--primary-color); padding: 4px 8px; border-radius: 4px; font-size: 12px;">Completed</span>
                            </div>
                            <p style="color: #64748b; font-size: 12px; margin: 0;">Feb 3, 2026 • $150 • 3 hours</p>
                            <p style="color: #64748b; margin: 10px 0 0 0;">Professional team cleaned every corner of your home. Excellent service!</p>
                        </div>

                        <div style="padding: 20px; background: var(--light-bg); border-left: 4px solid var(--primary-color); border-radius: 8px;">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 10px;">
                                <strong style="color: #0f172a;">Window Cleaning</strong>
                                <span style="background: rgba(16, 156, 61, 0.1); color: var(--primary-color); padding: 4px 8px; border-radius: 4px; font-size: 12px;">Completed</span>
                            </div>
                            <p style="color: #64748b; font-size: 12px; margin: 0;">Feb 1, 2026 • $80 • 1.5 hours</p>
                            <p style="color: #64748b; margin: 10px 0 0 0;">All windows cleaned inside and outside. Quick and efficient service.</p>
                        </div>

                        <div style="padding: 20px; background: var(--light-bg); border-left: 4px solid var(--primary-color); border-radius: 8px;">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 10px;">
                                <strong style="color: #0f172a;">Carpet Cleaning</strong>
                                <span style="background: rgba(16, 156, 61, 0.1); color: var(--primary-color); padding: 4px 8px; border-radius: 4px; font-size: 12px;">Completed</span>
                            </div>
                            <p style="color: #64748b; font-size: 12px; margin: 0;">Jan 15, 2026 • $120 • 2 hours</p>
                            <p style="color: #64748b; margin: 10px 0 0 0;">Deep carpet cleaning with stain removal. Carpets look brand new!</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- TAB 4: Profile -->
            <div id="profile" class="tab-content">
                <div class="content-section">
                    <h2 style="margin-bottom: 25px;">Edit Profile</h2>
                    <form>
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                            <div class="form-group">
                                <label>First Name</label>
                                <input type="text" class="form-control" value="John" required>
                            </div>
                            <div class="form-group">
                                <label>Last Name</label>
                                <input type="text" class="form-control" value="Doe" required>
                            </div>
                            <div class="form-group">
                                <label>Email Address</label>
                                <input type="email" class="form-control" value="john@example.com" required>
                            </div>
                            <div class="form-group">
                                <label>Phone Number</label>
                                <input type="tel" class="form-control" value="+1 (555) 123-4567" required>
                            </div>
                            <div class="form-group">
                                <label>City</label>
                                <input type="text" class="form-control" value="New York" required>
                            </div>
                            <div class="form-group">
                                <label>Zip Code</label>
                                <input type="text" class="form-control" value="10001" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Address</label>
                            <input type="text" class="form-control" value="123 Main Street" required>
                        </div>
                        <div class="form-group">
                            <label>Additional Notes</label>
                            <textarea class="form-control" rows="4" placeholder="Any special instructions for our cleaners..."></textarea>
                        </div>
                        <button type="submit" class="btn-primary-custom">Save Changes</button>
                    </form>
                </div>
            </div>

            <!-- TAB 5: Reviews -->
            <div id="reviews" class="tab-content">
                <div class="content-section">
                    <h2 style="margin-bottom: 25px;">My Reviews</h2>
                    
                    <div style="display: flex; flex-direction: column; gap: 20px;">
                        <div style="padding: 20px; border: 1px solid var(--border-color); border-radius: 8px;">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                                <div>
                                    <strong style="color: #0f172a;">Home Deep Cleaning</strong>
                                    <p style="color: #64748b; font-size: 12px; margin: 5px 0 0 0;">Feb 3, 2026</p>
                                </div>
                                <div style="color: #fbbf24; font-size: 16px;">
                                    ★★★★★
                                </div>
                            </div>
                            <p style="color: #64748b; margin: 0;">Excellent service! The team was professional, punctual, and thorough. Our home has never looked better. Highly recommended!</p>
                        </div>

                        <div style="padding: 20px; border: 1px solid var(--border-color); border-radius: 8px;">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                                <div>
                                    <strong style="color: #0f172a;">Window Cleaning</strong>
                                    <p style="color: #64748b; font-size: 12px; margin: 5px 0 0 0;">Feb 1, 2026</p>
                                </div>
                                <div style="color: #fbbf24; font-size: 16px;">
                                    ★★★★☆
                                </div>
                            </div>
                            <p style="color: #64748b; margin: 0;">Very good service. Fast and efficient. Windows are crystal clear. One minor issue but overall satisfied.</p>
                        </div>

                        <div style="padding: 20px; border: 1px solid var(--border-color); border-radius: 8px;">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                                <div>
                                    <strong style="color: #0f172a;">Carpet Cleaning</strong>
                                    <p style="color: #64748b; font-size: 12px; margin: 5px 0 0 0;">Jan 15, 2026</p>
                                </div>
                                <div style="color: #fbbf24; font-size: 16px;">
                                    ★★★★★
                                </div>
                            </div>
                            <p style="color: #64748b; margin: 0;">Amazing results! Stains that I thought were permanent are completely gone. Great value for money.</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="assets/js/vendor/jquery-3.7.1.min.js"></script>
    <script src="assets/js/bootstrap.min.js"></script>
    <script>
        // Tab Navigation
        document.querySelectorAll('.menu-link').forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                const tabName = this.getAttribute('data-tab');
                
                // Update active menu item
                document.querySelectorAll('.menu-link').forEach(el => el.classList.remove('active'));
                this.classList.add('active');
                
                // Update page title
                const titles = {
                    'overview': 'Welcome Back!',
                    'bookings': 'My Bookings',
                    'history': 'Service History',
                    'profile': 'Edit Profile',
                    'reviews': 'My Reviews'
                };
                document.getElementById('pageTitle').textContent = titles[tabName] || 'Dashboard';
                
                // Hide all tabs
                document.querySelectorAll('.tab-content').forEach(tab => {
                    tab.classList.remove('active');
                });
                
                // Show selected tab
                document.getElementById(tabName).classList.add('active');
                
                // Close sidebar on mobile
                if (window.innerWidth <= 768) {
                    document.getElementById('sidebar').classList.remove('active');
                }
            });
        });

        // Toggle Sidebar on Mobile
        document.getElementById('toggleBtn').addEventListener('click', function() {
            document.getElementById('sidebar').classList.toggle('active');
        });

        // Show/hide toggle button based on screen size
        function updateToggleButton() {
            const toggleBtn = document.getElementById('toggleBtn');
            if (window.innerWidth <= 768) {
                toggleBtn.style.display = 'inline-block';
            } else {
                toggleBtn.style.display = 'none';
                document.getElementById('sidebar').classList.remove('active');
            }
        }

        window.addEventListener('resize', updateToggleButton);
        updateToggleButton();

        // Filter Bookings
        function filterBookings(status) {
            console.log('Filter bookings by status:', status);
            // Add your filter logic here
        }

        // Close sidebar when clicking outside on mobile
        document.addEventListener('click', function(event) {
            const sidebar = document.getElementById('sidebar');
            const toggleBtn = document.getElementById('toggleBtn');
            
            if (window.innerWidth <= 768 && 
                !sidebar.contains(event.target) && 
                !toggleBtn.contains(event.target) &&
                sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
            }
        });
    </script>
</body>
</html>
